<?php
/**
 * Malware scan notification model.
 *
 * @package WP_Defender\Model\Notification
 */

namespace WP_Defender\Model\Notification;

use WP_Defender\Model\Scan;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\IO;
use WP_Defender\Traits\Scan_Email_Template;
use WP_Defender\Traits\Scan_Upsell;
use WP_Defender\Model\Notification\Malware_Report;
use WP_Defender\Behavior\WPMUDEV;

/**
 * Class Malware_Notification
 */
class Malware_Notification extends \WP_Defender\Model\Notification {
	use IO;
	use Scan_Email_Template;
	use Scan_Upsell;

	/**
	 * Option name.
	 *
	 * @var string
	 */
	protected $table = 'wd_malware_scanning_notification';

	public const SLUG = 'malware-notification';

	/**
	 * Get default values.
	 *
	 * @return array
	 */
	public function get_default_values(): array {
		$template = $this->get_email_template();

		return array(
			'subject_issue_found'     => $template['found']['subject'],
			'subject_issue_not_found' => $template['not_found']['subject'],
			'subject_error'           => $template['error']['subject'],
			'content_issue_found'     => $template['found']['body'],
			'content_issue_not_found' => $template['not_found']['body'],
			'content_error'           => $template['error']['body'],
		);
	}

	/**
	 * Before load.
	 *
	 * @return void
	 */
	protected function before_load(): void {
		$default = array(
			'slug'                 => self::SLUG,
			'title'                => __( 'Malware Scanning - Notification', 'defender-security' ),
			'status'               => self::STATUS_DISABLED,
			'description'          => __( 'Get email notifications when Defender has finished manual malware scans.', 'defender-security' ),
			// @since 3.0.0 Fix 'Guest'-line.
			'in_house_recipients'  => is_user_logged_in() ? array( $this->get_default_user() ) : array(),
			'out_house_recipients' => array(),
			'type'                 => 'notification',
			'dry_run'              => false,
			'configs'              => array(
				'always_send' => false,
				'error_send'  => false,
				'template'    => $this->get_email_template(),
			),
		);
		$this->import( $default );
	}

	/**
	 * The email engine.
	 *
	 * @param Scan   $scan    The scan model.
	 * @param object $obj     The object.
	 * @param string $subject The email subject.
	 * @param string $content The email content.
	 * @param array  $issues  The issues.
	 */
	protected function email_engine( Scan $scan, $obj, $subject, $content, $issues ): void {
		$service = wd_di()->get( \WP_Defender\Component\Notification::class );
		foreach ( $obj->in_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $service );
		}
		foreach ( $obj->out_house_recipients as $user ) {
			if ( self::USER_SUBSCRIBED !== $user['status'] ) {
				continue;
			}
			$this->send_to_user( $user['email'], $user['name'], $subject, $content, $issues, $service );
		}
	}

	/**
	 * Send the email.
	 *
	 * @param Scan $scan The scan model.
	 *
	 * @return void
	 */
	public function send( Scan $scan ) {
		$object = $this;
		if ( $scan->is_automation ) {
			$object = wd_di()->get( Malware_Report::class );
			// If Malware Reporting is unchecked, then return.
			if ( self::STATUS_ACTIVE !== $object->status ) {
				return;
			}
			// @since 2.7.0 Remove 'dry_run'-condition.
		}
		$issues = $scan->get_issues( null, Scan_Item::STATUS_ACTIVE );
		/**
		 * When the Defender is unable to scan files for some reason and the 'error_send' option is checked.
		 * Possible reasons:
		 * - no connection to Scan API,
		 * - no checksums received from wp.org,
		 * - scan process stuck.
		 */
		if (
			isset( $object->configs['error_send'] ) && ! empty(
				$object->configs['error_send']
				&& in_array( $scan->status, array( Scan::STATUS_ERROR, Scan::STATUS_IDLE ), true )
			)
		) {
			$subject = $object->configs['template']['error']['subject'];
			$content = nl2br( $object->configs['template']['error']['body'] );

			return $this->email_engine( $scan, $object, $subject, $content, $issues );
		}

		// Sometimes value may be as empty string for disabled 'always_send'.
		if ( empty( $object->configs['always_send'] ) && 0 === count( $issues ) ) {
			return;
		}
		// For another case.
		$templates = count( $issues ) ? $object->configs['template']['found'] : $object->configs['template']['not_found'];
		$subject   = $templates['subject'];
		$content   = nl2br( $templates['body'] );

		return $this->email_engine( $scan, $object, $subject, $content, $issues );
	}

	/**
	 * Send email to user.
	 *
	 * @param string $email   The email.
	 * @param string $name    The name.
	 * @param string $subject The email subject.
	 * @param string $content The email content.
	 * @param array  $issues  The issues.
	 * @param object $service The service.
	 *
	 * @throws \DI\DependencyException Throws exception when the dependency is not resolved.
	 * @throws \DI\NotFoundException   Throws exception when the dependency is not found.
	 */
	private function send_to_user( $email, $name, $subject, $content, $issues, $service ) {
		$controller_scan = wd_di()->get( \WP_Defender\Controller\Scan::class );
		$replacers       = array(
			'issues_count' => is_array( $issues ) || $issues instanceof \Countable ? (string) count( $issues ) : '0',
			'issues_list'  => $controller_scan->render_partial(
				'email/scan-issue-list',
				array(
					'issues' => $issues,
					'email'  => $email,
					'upsell' => ( new WPMUDEV() )->is_pro() ? $this->get_scan_upsell( 'email' ) : [],
				),
				false
			),
			'site_url'     => network_site_url(),
			'user_name'    => $name,
		);

		/**
		 * Filters the replaced data. It's without replacement.
		 *
		 * @deprecated 3.2.0
		 * @param array $replacers Email params.
		 */
		$replacers = apply_filters_deprecated( 'wd_notification_email_params', array( $replacers ), '3.2.0' );

		$filter_name = ( is_array( $issues ) || $issues instanceof \Countable ? count( $issues ) : 0 ) > 0
			? 'wd_notification_email_subject_issue'
			: 'wd_notification_email_subject';
		$subject     = apply_filters( $filter_name, $subject );

		/**
		 * It's without replacement.
		 *
		 * @deprecated 3.2.0
		 */
		$content = apply_filters_deprecated(
			'wd_notification_email_content_before',
			array( $content ),
			'3.2.0'
		);
		foreach ( $replacers as $key => $replacer ) {

			if ( is_scalar( $replacer ) && ! is_string( $replacer ) ) {
				$replacer = (string) $replacer;
			}

			$content = str_replace( "{{$key}}", $replacer, $content );
			$content = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $content );
			$subject = str_replace( "{{$key}}", $replacer, $subject );
			$subject = str_replace( '{' . strtoupper( $key ) . '}', $replacer, $subject );
		}
		/**
		 * It's without replacement.
		 *
		 * @deprecated 3.2.0
		 */
		$content          = apply_filters_deprecated(
			'wd_notification_email_content_after',
			array( $content ),
			'3.2.0'
		);
		$unsubscribe_link = $service->create_unsubscribe_url( $this->slug, $email );
		$template         = $controller_scan->render_partial(
			'email/index',
			array(
				'title'            => __( 'Malware Scanning', 'defender-security' ),
				'content_body'     => sprintf( '<div style="font-size: 16px">%s</div>', $content ),
				'unsubscribe_link' => $unsubscribe_link,
			),
			false
		);

		$headers = wd_di()->get( \WP_Defender\Component\Mail::class )->get_headers(
			defender_noreply_email( 'wd_scan_noreply_email' ),
			self::SLUG
		);

		$ret = wp_mail( $email, $subject, $template, $headers );
		if ( $ret ) {
			$this->save_log( $email );
		}
	}

	/**
	 * Define settings labels.
	 *
	 * @return array
	 */
	public function labels(): array {
		return array(
			'notification'                  => __( 'Malware Scanning - Notification', 'defender-security' ),
			'always_send_notification'      => __( 'Send notifications when no issues are detected', 'defender-security' ),
			'error_send'                    => __( "Send notifications when Defender couldn't scan the files", 'defender-security' ),
			'email_subject_issue_found'     => __( 'Email subject when an issue is found', 'defender-security' ),
			'email_subject_issue_not_found' => __( 'Email subject when no issues are found', 'defender-security' ),
			'email_subject_error'           => __( 'Email subject when failed to scan', 'defender-security' ),
			'email_content_issue_found'     => __( 'Email body when an issue is found', 'defender-security' ),
			'email_content_issue_not_found' => __( 'Email body when no issues are found', 'defender-security' ),
			'email_content_error'           => __( 'When failed to scan', 'defender-security' ),
			'notification_subscribers'      => __( 'Recipients', 'defender-security' ),
		);
	}

	/**
	 * Try to send email if:
	 * Malware Scanning - Notification is enabled,
	 * Malware Scanning - Reporting is enabled or Scan model has is_automation = true.
	 *
	 * @return bool
	 */
	public function check_options(): bool {
		return self::STATUS_ACTIVE === $this->status
			|| self::STATUS_ACTIVE === wd_di()->get( Malware_Report::class )->status;
	}

	/**
	 * Additional converting rules.
	 *
	 * @param array $configs The configs.
	 *
	 * @return array
	 * @since 3.1.0
	 */
	public function type_casting( array $configs ): array {
		$configs['always_send'] = (bool) $configs['always_send'];
		$configs['error_send']  = (bool) $configs['error_send'];
		// Additional security layer because parent 'configs'-property doesn't have sanitized stuff for nested properties.
		$template                                    = $configs['template'];
		$configs['template']['found']['subject']     = sanitize_text_field( $template['found']['subject'] );
		$configs['template']['found']['body']        = sanitize_textarea_field( $template['found']['body'] );
		$configs['template']['not_found']['subject'] = sanitize_text_field( $template['not_found']['subject'] );
		$configs['template']['not_found']['body']    = sanitize_textarea_field( $template['not_found']['body'] );
		$configs['template']['error']['subject']     = sanitize_text_field( $template['error']['subject'] );
		$configs['template']['error']['body']        = sanitize_textarea_field( $template['error']['body'] );

		return $configs;
	}
}
